<?php defined('BASEPATH') OR exit('No direct script access allowed');

class PembayaranSiswa_model extends CI_Model
{
    // Samakan dengan nama tabel di MySQL (huruf kecil)
    private $table = 'pembayaransiswa';

    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    /* ===================== Helpers ===================== */

    /** Cek keberadaan kolom (case-insensitive) */
    private function has_col(string $col): bool
    {
        $cols = array_map('strtolower', $this->db->list_fields($this->table));
        return in_array(strtolower($col), $cols, true);
    }

    /** Nama kolom 'jenis' yang tersedia, atau null jika tak ada */
    private function jenis_col(): ?string
    {
        if ($this->has_col('JenisPembayaran')) return 'JenisPembayaran';
        if ($this->has_col('Jenis'))            return 'Jenis';
        return null;
    }

    /** Apakah kolom StatusKonfirmasi bertipe numerik? */
    private function status_is_numeric(): bool
    {
        if (!$this->has_col('StatusKonfirmasi')) return false;

        // int/tinyint/… dianggap numerik
        $row = $this->db->query(
            "SELECT DATA_TYPE
               FROM information_schema.COLUMNS
              WHERE TABLE_SCHEMA = DATABASE()
                AND TABLE_NAME   = ?
                AND COLUMN_NAME  = 'StatusKonfirmasi'
              LIMIT 1",
            [$this->table]
        )->row();

        if (!$row) return false;
        $t = strtolower((string)$row->DATA_TYPE);
        return in_array($t, ['int','integer','tinyint','smallint','mediumint','bigint','decimal','numeric']);
    }

    /** Nilai 'approved' mengikuti tipe kolom */
    private function approved_value()
    {
        return $this->status_is_numeric() ? 1 : 'Approved';
    }

    /** Ambil properti dari $row dengan fallback huruf kecil/besar */
    private function rget($row, string $a, string $b = null)
    {
        $lo = strtolower($a);
        $v1 = $row->$a ?? null;
        if ($v1 !== null) return $v1;
        if ($b && isset($row->$b)) return $row->$b;
        return $row->$lo ?? null;
    }

    /* ===================== Public API ===================== */

    /**
     * Insert baris ledger saat admin approve.
     * $pembayaran_row wajib membawa: PembayaranID, SiswaID, **UserID siswa**, JumlahBayar, BuktiTransfer (opsional).
     * $admin_user_id dipakai ke kolom ApprovedBy bila ada (bukan ke UserID!).
     */
    public function insert_from_pembayaran($pembayaran_row, int $admin_user_id = 0, ?string $jenis = null)
    {
        if (!$pembayaran_row) return false;

        // Normalisasi field dari sumber (tahan huruf kecil/besar)
        $pembayaran_id = $this->rget($pembayaran_row, 'PembayaranID');
        $siswa_id      = $this->rget($pembayaran_row, 'SiswaID');
        $siswa_user_id = (int) $this->rget($pembayaran_row, 'UserID'); // ← FK ke siswa(UserID)
        $nominal       = (float)($this->rget($pembayaran_row, 'JumlahBayar') ?? 0);
        $bukti         = $this->rget($pembayaran_row, 'BuktiTransfer');

        if (!$pembayaran_id) return false;

        $data = [
            'PembayaranID'  => $pembayaran_id,
            'SiswaID'       => $siswa_id,
            'Nominal'       => $nominal,
            'BuktiTransfer' => $bukti ?: null,
        ];

        // Kolom status (bila ada)
        if ($this->has_col('StatusKonfirmasi')) {
            $data['StatusKonfirmasi'] = $this->approved_value();
        }

        // Kolom UserID pada ledger HARUS diisi UserID siswa agar FK ke siswa(UserID) aman
        if ($this->has_col('UserID')) {
            $data['UserID'] = $siswa_user_id;
        }

        // Jenis (Pelatihan/Raftel) jika kolom tersedia
        $jenisCol = $this->jenis_col();
        if ($jenisCol !== null && $jenis !== null) {
            $data[$jenisCol] = $jenis;
        }

        // Tanggal/metadata yang tersedia
        $now = date('Y-m-d H:i:s');
        if ($this->has_col('TanggalBayar')) $data['TanggalBayar'] = $now;
        if ($this->has_col('CreatedAt'))    $data['CreatedAt']    = $now;
        if ($this->has_col('UpdatedAt'))    $data['UpdatedAt']    = $now;
        if ($this->has_col('ApprovedAt'))   $data['ApprovedAt']   = $now;
        if ($this->has_col('ApprovedBy'))   $data['ApprovedBy']   = $admin_user_id; // simpan admin di kolom ini kalau ada

        $this->db->insert($this->table, $data);
        return $this->db->insert_id();
    }

    /** Total nominal approved untuk satu siswa (opsional filter jenis) */
    public function sum_nominal_by_siswa($siswa_id, $jenis = null): float
    {
        $this->db->select_sum('Nominal')
                 ->from($this->table)
                 ->where('SiswaID', $siswa_id);

        if ($this->has_col('StatusKonfirmasi')) {
            $this->db->where('StatusKonfirmasi', $this->approved_value());
        }

        $jenisCol = $this->jenis_col();
        if ($jenisCol !== null && $jenis !== null) {
            $this->db->where($jenisCol, $jenis);
        }

        $row = $this->db->get()->row();
        return $row ? (float)$row->Nominal : 0.0;
    }

    /** Alias kompatibilitas lama */
    public function sum_nominal_by_siswaid($siswa_id, $jenis = null): float
    {
        return $this->sum_nominal_by_siswa($siswa_id, $jenis);
    }

    /** Ambil ledger per siswa, terbaru di atas (opsional filter jenis) */
    public function get_ledger_by_siswa($siswa_id, $jenis = null)
    {
        $this->db->from($this->table)->where('SiswaID', $siswa_id);

        $jenisCol = $this->jenis_col();
        if ($jenisCol !== null && $jenis !== null) {
            $this->db->where($jenisCol, $jenis);
        }

        if ($this->has_col('TanggalBayar')) {
            $this->db->order_by('TanggalBayar', 'DESC');
        } elseif ($this->has_col('CreatedAt')) {
            $this->db->order_by('CreatedAt', 'DESC');
        }

        return $this->db->get()->result();
    }
}
